"use strict";
(() => {
  // src/helpers/settings.ts
  function getSettingsSummary(settings, reference) {
    const hasToken = Boolean(settings.authToken.trim());
    const hasOnBehalfOf = Boolean(settings.onBehalfOf);
    const isConfigured = hasToken && hasOnBehalfOf;
    if (isConfigured && reference.status === "ready") {
      const user = reference.users.find((u) => String(u.id) === settings.onBehalfOf);
      const userName = user?.name || "Unknown user";
      return { text: `Connected as ${userName}`, visible: true };
    }
    if (hasToken && reference.status === "loading") {
      return { text: "Connecting...", visible: true };
    }
    return { text: "", visible: false };
  }
  function normalizeBaseUrl(url) {
    return url.replace(/\/$/, "");
  }

  // src/panel.ts
  var DEFAULT_WORKER_BASE_URL = "https://outreach-relay-worker.ewa-services-com.workers.dev";
  var state = {
    extraction: {
      status: "idle",
      supported: false,
      loggedIn: true,
      onLinkedIn: false,
      reason: "",
      missing: [],
      messages: []
    },
    reference: {
      status: "idle",
      jobs: [],
      sources: [],
      users: [],
      error: null
    },
    form: {
      candidate: {
        firstName: "",
        lastName: "",
        headline: "",
        company: "",
        location: "",
        email: "",
        phone: ""
      },
      outreach: {
        sourceSurface: "",
        linkedinUrl: "",
        participants: "",
        messageSnippet: ""
      },
      application: {
        jobId: "",
        sourceId: "",
        recruiterId: ""
      },
      extracted: {
        emails: [],
        phones: [],
        messages: [],
        includeMessages: true,
        includeMessagesTouched: false
      }
    },
    settings: {
      workerBaseUrl: DEFAULT_WORKER_BASE_URL,
      authToken: "",
      onBehalfOf: ""
    },
    submission: {
      status: "idle",
      error: null,
      result: null,
      requestId: null
    },
    validationErrors: {},
    backendErrors: {},
    showErrors: false,
    touched: /* @__PURE__ */ new Set()
  };
  var EXTRACTION_RETRY_LIMIT = 3;
  var EXTRACTION_RETRY_DELAY_MS = 1200;
  var TAB_CHANGE_DEBOUNCE_MS = 500;
  var extractionRetryCount = 0;
  var extractionRetryTimeout = null;
  var tabChangeDebounceTimeout = null;
  var dom = {
    statusBadge: document.querySelector("[data-status-badge]"),
    statusText: document.querySelector("[data-status-text]"),
    statusDetail: document.querySelector("[data-status-detail]"),
    statusAlerts: document.getElementById("status-alerts"),
    errorSummary: document.getElementById("error-summary"),
    errorList: document.getElementById("error-list"),
    submitButton: document.getElementById("submit-button"),
    refreshButton: document.getElementById("refresh-extraction"),
    retryReference: document.getElementById("retry-reference"),
    retryApplication: document.getElementById("retry-application"),
    submissionStatus: document.getElementById("submission-status"),
    resultCard: document.getElementById("result-card"),
    resultLinks: document.getElementById("result-links"),
    referenceStatus: document.getElementById("reference-status"),
    emailOptions: document.getElementById("email-options"),
    phoneOptions: document.getElementById("phone-options"),
    emailDetected: document.getElementById("email-detected"),
    phoneDetected: document.getElementById("phone-detected"),
    messageList: document.getElementById("message-list"),
    messageEmpty: document.getElementById("message-empty"),
    includeMessages: document.getElementById("include-messages"),
    jobSelect: document.getElementById("job-select"),
    sourceSelect: document.getElementById("source-select"),
    recruiterSelect: document.getElementById("recruiter-select"),
    onBehalfOfSelect: document.getElementById("on-behalf-select"),
    fieldInputs: Array.from(document.querySelectorAll("[data-field]")),
    errorFields: Array.from(document.querySelectorAll("[data-error-for]")),
    toggleButtons: Array.from(document.querySelectorAll("[data-toggle]")),
    pasteTokenBtn: document.getElementById("paste-token-btn"),
    pasteIcon: document.getElementById("paste-icon"),
    settingsSummary: document.getElementById("settings-summary"),
    authTokenInput: document.getElementById("auth-token-input"),
    workerUrlInput: document.getElementById("worker-url-input"),
    workerUrlStatus: document.getElementById("worker-url-status")
  };
  var STORAGE_KEYS = {
    settings: "settings",
    draft: "formDraft",
    lastJobId: "lastJobId",
    lastSourceId: "lastSourceId",
    onBehalfOf: "onBehalfOf",
    referenceData: "referenceData",
    collapsedSections: "collapsedSections"
  };
  init();
  async function init() {
    bindFieldListeners();
    bindActionListeners();
    bindRuntimeListeners();
    bindOnlineListeners();
    await loadCollapsedSections();
    await loadStoredSettings();
    await loadStoredDraft();
    await loadSubmissionState();
    await refreshExtraction();
    await fetchReferenceData();
    render();
  }
  function bindFieldListeners() {
    dom.fieldInputs.forEach((input) => {
      const handler = (event) => {
        const target = event.target;
        const key = target.dataset.field;
        if (key) {
          updateField(key, target.value, { persist: true });
        }
      };
      input.addEventListener("input", handler);
      input.addEventListener("change", handler);
    });
  }
  function bindActionListeners() {
    dom.refreshButton.addEventListener("click", () => refreshExtraction());
    dom.retryReference.addEventListener("click", () => fetchReferenceData());
    dom.submitButton.addEventListener("click", () => submitPayload());
    dom.retryApplication.addEventListener("click", () => retryApplication());
    dom.includeMessages.addEventListener("change", (event) => {
      state.form.extracted.includeMessages = event.target.checked;
      state.form.extracted.includeMessagesTouched = true;
      renderMessages();
    });
    dom.toggleButtons.forEach((button) => {
      button.addEventListener("click", () => {
        const toggleId = button.dataset.toggle;
        if (toggleId) {
          toggleSection(toggleId);
        }
      });
    });
    dom.pasteTokenBtn.addEventListener("click", () => pasteFromClipboard());
    dom.workerUrlInput.addEventListener("blur", () => validateWorkerUrl());
    dom.workerUrlInput.addEventListener("change", () => validateWorkerUrl());
  }
  async function pasteFromClipboard() {
    try {
      const text = await navigator.clipboard.readText();
      if (text) {
        updateField("settings.authToken", text.trim(), { persist: true });
        dom.authTokenInput.value = text.trim();
        showPasteFeedback();
      }
    } catch {
      dom.authTokenInput.focus();
      document.execCommand("paste");
    }
  }
  function showPasteFeedback() {
    const originalText = dom.pasteIcon.textContent;
    dom.pasteIcon.textContent = "\u2713";
    dom.pasteTokenBtn.style.background = "rgba(26, 181, 105, 0.15)";
    setTimeout(() => {
      dom.pasteIcon.textContent = originalText;
      dom.pasteTokenBtn.style.background = "";
    }, 1500);
  }
  var urlValidationAbort = null;
  async function validateWorkerUrl() {
    const url = dom.workerUrlInput.value.trim();
    dom.workerUrlStatus.className = "input-status";
    dom.workerUrlInput.classList.remove("input--valid");
    if (!url) {
      return;
    }
    try {
      new URL(url);
    } catch {
      return;
    }
    if (urlValidationAbort) {
      urlValidationAbort.abort();
    }
    urlValidationAbort = new AbortController();
    dom.workerUrlStatus.className = "input-status visible input-status--loading";
    try {
      const healthUrl = `${normalizeBaseUrl(url)}/health`;
      const response = await fetch(healthUrl, {
        signal: urlValidationAbort.signal,
        mode: "cors"
      });
      if (response.ok) {
        dom.workerUrlStatus.className = "input-status visible input-status--valid";
        dom.workerUrlInput.classList.add("input--valid");
      } else {
        dom.workerUrlStatus.className = "input-status visible input-status--invalid";
      }
    } catch (error) {
      if (error instanceof Error && error.name === "AbortError") {
        return;
      }
      dom.workerUrlStatus.className = "input-status visible input-status--invalid";
    }
  }
  async function toggleSection(sectionId) {
    const section = document.getElementById(sectionId);
    if (!section) return;
    const isCollapsed = section.classList.toggle("card--collapsed");
    const data = await chrome.storage.local.get(STORAGE_KEYS.collapsedSections);
    const collapsed = data[STORAGE_KEYS.collapsedSections] || {};
    collapsed[sectionId] = isCollapsed;
    chrome.storage.local.set({ [STORAGE_KEYS.collapsedSections]: collapsed });
  }
  async function loadCollapsedSections() {
    const data = await chrome.storage.local.get(STORAGE_KEYS.collapsedSections);
    const collapsed = data[STORAGE_KEYS.collapsedSections] || {};
    Object.entries(collapsed).forEach(([sectionId, isCollapsed]) => {
      const section = document.getElementById(sectionId);
      if (section) {
        section.classList.toggle("card--collapsed", isCollapsed);
      }
    });
  }
  function bindRuntimeListeners() {
    chrome.runtime.onMessage.addListener((message) => {
      if (!message || typeof message !== "object") {
        return;
      }
      if (message.type === "submission-updated") {
        applySubmissionState(message.state);
      }
      if (message.type === "tab-changed") {
        debouncedRefreshExtraction();
      }
    });
  }
  function debouncedRefreshExtraction() {
    if (tabChangeDebounceTimeout) {
      clearTimeout(tabChangeDebounceTimeout);
    }
    tabChangeDebounceTimeout = setTimeout(() => {
      tabChangeDebounceTimeout = null;
      refreshExtraction();
    }, TAB_CHANGE_DEBOUNCE_MS);
  }
  function bindOnlineListeners() {
    window.addEventListener("online", () => {
      render();
    });
    window.addEventListener("offline", () => {
      render();
    });
  }
  async function loadStoredSettings() {
    const data = await chrome.storage.local.get([STORAGE_KEYS.settings, STORAGE_KEYS.onBehalfOf, STORAGE_KEYS.collapsedSections]);
    if (data[STORAGE_KEYS.settings]) {
      const { workerBaseUrl, authToken, onBehalfOf } = data[STORAGE_KEYS.settings];
      updateField("settings.workerBaseUrl", workerBaseUrl || DEFAULT_WORKER_BASE_URL, { markTouched: false, persist: false });
      updateField("settings.authToken", authToken || "", { markTouched: false, persist: false });
      const storedOnBehalfOf = data[STORAGE_KEYS.onBehalfOf] || onBehalfOf || "";
      updateField("settings.onBehalfOf", storedOnBehalfOf, { markTouched: false, persist: false });
      const collapsed = data[STORAGE_KEYS.collapsedSections] || {};
      if (collapsed["settings-card"] === void 0 && workerBaseUrl && authToken && storedOnBehalfOf) {
        const settingsCard = document.getElementById("settings-card");
        if (settingsCard) settingsCard.classList.add("card--collapsed");
      }
    } else {
      updateField("settings.workerBaseUrl", DEFAULT_WORKER_BASE_URL, { markTouched: false, persist: true });
      if (data[STORAGE_KEYS.onBehalfOf]) {
        updateField("settings.onBehalfOf", data[STORAGE_KEYS.onBehalfOf], { markTouched: false, persist: false });
      }
    }
    validateWorkerUrl();
  }
  async function loadStoredDraft() {
    const data = await chrome.storage.local.get([STORAGE_KEYS.draft, STORAGE_KEYS.lastJobId, STORAGE_KEYS.lastSourceId]);
    if (data[STORAGE_KEYS.draft]) {
      hydrateDraft(data[STORAGE_KEYS.draft]);
    }
    if (data[STORAGE_KEYS.lastJobId]) {
      updateField("application.jobId", data[STORAGE_KEYS.lastJobId], { markTouched: false, persist: false });
    }
    if (data[STORAGE_KEYS.lastSourceId]) {
      updateField("application.sourceId", data[STORAGE_KEYS.lastSourceId], { markTouched: false, persist: false });
    }
  }
  async function loadSubmissionState() {
    const response = await sendMessage({ type: "get-submission-state" });
    if (response && response.state) {
      applySubmissionState(response.state);
    }
  }
  function applySubmissionState(submissionState) {
    state.submission = {
      status: submissionState.status,
      error: submissionState.error,
      result: submissionState.result,
      requestId: submissionState.requestId ?? null
    };
    if (submissionState?.error?.fields) {
      state.backendErrors = { ...submissionState.error.fields };
      state.showErrors = true;
    }
    render();
  }
  function hydrateDraft(draft) {
    if (!draft) {
      return;
    }
    Object.entries(draft).forEach(([key, value]) => {
      const shouldMark = value !== void 0 && value !== null && String(value).trim() !== "";
      updateField(key, value, { markTouched: shouldMark, persist: false });
    });
  }
  function updateField(key, value, { markTouched = true, persist = true } = {}) {
    if (!key) {
      return;
    }
    if (markTouched) {
      state.touched.add(key);
    }
    if (state.backendErrors[key]) {
      delete state.backendErrors[key];
    }
    setFieldValue(key, value);
    if (persist) {
      persistDraft();
      if (key.startsWith("settings.")) {
        persistSettings();
        if (key === "settings.workerBaseUrl" || key === "settings.authToken") {
          fetchReferenceData();
        }
      }
      if (key === "application.jobId") {
        chrome.storage.local.set({ [STORAGE_KEYS.lastJobId]: value });
      }
      if (key === "application.sourceId") {
        chrome.storage.local.set({ [STORAGE_KEYS.lastSourceId]: value });
      }
      if (key === "settings.onBehalfOf") {
        chrome.storage.local.set({ [STORAGE_KEYS.onBehalfOf]: value });
      }
    }
    render();
  }
  function setFieldValue(key, value) {
    const [section, field] = key.split(".");
    if (!section || !field) {
      return;
    }
    if (section === "settings") {
      state.settings[field] = value;
    } else if (section === "candidate") {
      state.form.candidate[field] = value;
    } else if (section === "outreach") {
      state.form.outreach[field] = value;
    } else if (section === "application") {
      state.form.application[field] = value;
    }
    const input = dom.fieldInputs.find((element) => element.dataset.field === key);
    if (input && input.value !== value) {
      input.value = value;
    }
  }
  async function persistSettings() {
    const payload = {
      workerBaseUrl: state.settings.workerBaseUrl.trim(),
      authToken: state.settings.authToken.trim(),
      onBehalfOf: state.settings.onBehalfOf
    };
    await chrome.storage.local.set({ [STORAGE_KEYS.settings]: payload });
  }
  async function persistDraft() {
    const draft = {
      "candidate.firstName": state.form.candidate.firstName,
      "candidate.lastName": state.form.candidate.lastName,
      "candidate.headline": state.form.candidate.headline,
      "candidate.company": state.form.candidate.company,
      "candidate.location": state.form.candidate.location,
      "candidate.email": state.form.candidate.email,
      "candidate.phone": state.form.candidate.phone,
      "outreach.sourceSurface": state.form.outreach.sourceSurface,
      "outreach.linkedinUrl": state.form.outreach.linkedinUrl,
      "outreach.participants": state.form.outreach.participants,
      "outreach.messageSnippet": state.form.outreach.messageSnippet,
      "application.jobId": state.form.application.jobId,
      "application.sourceId": state.form.application.sourceId,
      "application.recruiterId": state.form.application.recruiterId
    };
    await chrome.storage.local.set({ [STORAGE_KEYS.draft]: draft });
  }
  async function refreshExtraction({ preserveRetry = false } = {}) {
    if (!preserveRetry) {
      resetExtractionRetry();
      resetExtractionData({ clearFields: true });
    }
    state.extraction.status = "loading";
    renderStatus();
    const response = await sendMessage({ type: "request-extraction" });
    if (!response || response.error) {
      const reason = response?.error || "Extraction failed";
      const retrying = scheduleExtractionRetry(reason);
      const supported = response?.supported !== false;
      const loggedIn = response?.loggedIn !== false;
      const onLinkedIn = response?.onLinkedIn === true;
      if (!onLinkedIn) {
        state.extraction.status = "idle";
      } else if (!loggedIn) {
        state.extraction.status = "warning";
      } else if (!supported) {
        state.extraction.status = "warning";
      } else if (retrying) {
        state.extraction.status = "loading";
      } else {
        state.extraction.status = "warning";
      }
      state.extraction.supported = supported;
      state.extraction.loggedIn = loggedIn;
      state.extraction.onLinkedIn = onLinkedIn;
      state.extraction.reason = retrying ? `Waiting for LinkedIn data... (${extractionRetryCount}/${EXTRACTION_RETRY_LIMIT})` : reason;
      state.extraction.missing = [];
      state.extraction.messages = [];
      renderStatus();
      return;
    }
    resetExtractionRetry();
    state.extraction.supported = Boolean(response.supported);
    state.extraction.loggedIn = response.loggedIn !== false;
    state.extraction.onLinkedIn = true;
    state.extraction.reason = response.reason || "";
    state.extraction.missing = response.missingFields || [];
    state.extraction.messages = response.outreach?.messages || [];
    if (!state.extraction.loggedIn) {
      state.extraction.status = "warning";
    } else if (!state.extraction.supported) {
      state.extraction.status = "warning";
    } else {
      state.extraction.status = response.partial ? "warning" : "ready";
    }
    applyExtractionToForm(response);
    render();
  }
  function resetExtractionData({ clearFields = false } = {}) {
    state.extraction.reason = "";
    state.extraction.missing = [];
    state.extraction.messages = [];
    state.form.extracted.emails = [];
    state.form.extracted.phones = [];
    state.form.extracted.messages = [];
    if (!state.form.extracted.includeMessagesTouched) {
      state.form.extracted.includeMessages = false;
    }
    if (clearFields) {
      clearFieldsIfUntouched([
        "candidate.firstName",
        "candidate.lastName",
        "candidate.headline",
        "candidate.company",
        "candidate.location",
        "candidate.email",
        "candidate.phone",
        "outreach.sourceSurface",
        "outreach.linkedinUrl",
        "outreach.participants",
        "outreach.messageSnippet"
      ]);
    }
  }
  function clearFieldsIfUntouched(keys) {
    keys.forEach((key) => {
      if (!state.touched.has(key)) {
        setFieldValue(key, "");
      }
    });
  }
  function scheduleExtractionRetry(reason) {
    if (!isRetryableExtractionError(reason) || extractionRetryCount >= EXTRACTION_RETRY_LIMIT) {
      return false;
    }
    extractionRetryCount += 1;
    if (extractionRetryTimeout) {
      clearTimeout(extractionRetryTimeout);
    }
    extractionRetryTimeout = setTimeout(() => {
      refreshExtraction({ preserveRetry: true });
    }, EXTRACTION_RETRY_DELAY_MS);
    return true;
  }
  function resetExtractionRetry() {
    if (extractionRetryTimeout) {
      clearTimeout(extractionRetryTimeout);
      extractionRetryTimeout = null;
    }
    extractionRetryCount = 0;
  }
  function isRetryableExtractionError(reason) {
    if (!reason || typeof reason !== "string") {
      return false;
    }
    return /Receiving end does not exist|Could not establish connection|message port closed|No response/i.test(reason);
  }
  function applyExtractionToForm(response) {
    if (!response || !response.candidate) {
      return;
    }
    const candidate = response.candidate || {};
    const outreach = response.outreach || {};
    applyIfUntouched("candidate.firstName", candidate.firstName || "");
    applyIfUntouched("candidate.lastName", candidate.lastName || "");
    applyIfUntouched("candidate.headline", candidate.headline || "");
    applyIfUntouched("candidate.company", candidate.company || "");
    applyIfUntouched("candidate.location", candidate.location || "");
    const extractedEmails = candidate.emails || [];
    const extractedPhones = candidate.phones || [];
    const extractedMessages = outreach.messages || [];
    state.form.extracted.emails = extractedEmails;
    state.form.extracted.phones = extractedPhones;
    state.form.extracted.messages = extractedMessages;
    if (!state.form.extracted.includeMessagesTouched) {
      state.form.extracted.includeMessages = extractedMessages.length > 0;
    } else if (!extractedMessages.length) {
      state.form.extracted.includeMessages = false;
    }
    if (!state.form.candidate.email && extractedEmails.length === 1) {
      applyIfUntouched("candidate.email", extractedEmails[0].value || "");
    }
    if (!state.form.candidate.phone && extractedPhones.length === 1) {
      applyIfUntouched("candidate.phone", extractedPhones[0].value || "");
    }
    applyIfUntouched("outreach.sourceSurface", outreach.sourceSurface || "");
    applyIfUntouched("outreach.linkedinUrl", outreach.linkedinUrl || "");
    applyIfUntouched("outreach.participants", (outreach.participants || []).join(", "));
    if (!state.touched.has("outreach.messageSnippet")) {
      const snippet = outreach.messageSnippet || buildMessageSnippet(outreach.messages || []);
      applyIfUntouched("outreach.messageSnippet", snippet);
    }
  }
  function applyIfUntouched(key, value) {
    if (!state.touched.has(key)) {
      updateField(key, value, { markTouched: false, persist: false });
    }
  }
  function buildMessageSnippet(messages) {
    if (!messages || !messages.length) {
      return "";
    }
    return messages.slice(-3).map((message) => {
      const direction = message.direction === "outbound" ? "You" : message.sender || "Them";
      const timestamp = message.timestamp ? ` (${message.timestamp})` : "";
      return `${direction}${timestamp}: ${message.text}`;
    }).join("\n");
  }
  function applyReferenceDefaults() {
    if (!state.form.application.sourceId && state.reference.sources.length) {
      const linkedInSource = state.reference.sources.find((s) => s.name === "LinkedIn Prospecting");
      if (linkedInSource) {
        state.form.application.sourceId = String(linkedInSource.id);
      }
    }
    if (!state.form.application.recruiterId && state.settings.onBehalfOf) {
      state.form.application.recruiterId = state.settings.onBehalfOf;
    }
  }
  async function fetchReferenceData() {
    if (!state.settings.workerBaseUrl || !state.settings.authToken) {
      state.reference.status = "idle";
      state.reference.error = "Add worker settings to load reference data.";
      renderReferenceData();
      return;
    }
    const cached = await chrome.storage.local.get(STORAGE_KEYS.referenceData);
    if (cached[STORAGE_KEYS.referenceData]) {
      const { jobs, sources, users } = cached[STORAGE_KEYS.referenceData];
      state.reference.jobs = jobs || [];
      state.reference.sources = sources || [];
      state.reference.users = users || [];
      state.reference.status = "ready";
      applyReferenceDefaults();
      render();
    } else {
      state.reference.status = "loading";
      state.reference.error = null;
      render();
    }
    try {
      const response = await fetch(`${normalizeBaseUrl(state.settings.workerBaseUrl)}/api/reference-data`, {
        headers: {
          Authorization: `Bearer ${state.settings.authToken.trim()}`
        }
      });
      if (!response.ok) {
        const data2 = await response.json().catch(() => ({}));
        if (response.status === 401) {
          throw new Error("Unauthorized. Check the worker auth token.");
        }
        throw new Error(data2?.error?.message || `Reference data failed (${response.status})`);
      }
      const data = await response.json();
      state.reference.jobs = data.jobs || [];
      state.reference.sources = data.sources || [];
      state.reference.users = data.users || [];
      state.reference.status = "ready";
      chrome.storage.local.set({
        [STORAGE_KEYS.referenceData]: {
          jobs: state.reference.jobs,
          sources: state.reference.sources,
          users: state.reference.users
        }
      });
      applyReferenceDefaults();
      render();
    } catch (error) {
      if (!state.reference.jobs.length && !state.reference.sources.length && !state.reference.users.length) {
        state.reference.status = "error";
        state.reference.error = error.message;
      }
      render();
    }
  }
  async function submitPayload() {
    if (state.submission.status === "in_progress") {
      return;
    }
    state.showErrors = true;
    if (!validateForm()) {
      renderErrors();
      return;
    }
    const payload = buildPayload();
    state.submission.status = "in_progress";
    state.submission.error = null;
    state.submission.result = null;
    renderSubmission();
    const response = await sendMessage({
      type: "submit-payload",
      payload,
      settings: {
        workerBaseUrl: normalizeBaseUrl(state.settings.workerBaseUrl),
        authToken: state.settings.authToken.trim()
      }
    });
    if (!response) {
      state.submission.status = "error";
      state.submission.error = { message: "No response from background worker." };
    } else if (response.error) {
      state.submission.status = "error";
      state.submission.error = response.error;
      state.submission.requestId = response.requestId || null;
      if (response.error.fields) {
        state.backendErrors = { ...response.error.fields };
        state.showErrors = true;
      }
    } else if (response.result) {
      state.submission.requestId = response.requestId || response.result.requestId || null;
      state.backendErrors = {};
    }
    render();
  }
  async function retryApplication() {
    if (state.submission.status === "in_progress") {
      return;
    }
    const candidateId = state.submission.result?.candidateId;
    if (!candidateId) {
      return;
    }
    state.showErrors = true;
    if (!validateForm()) {
      renderErrors();
      return;
    }
    const payload = buildPayload({ candidateId });
    state.submission.status = "in_progress";
    state.submission.error = null;
    renderSubmission();
    const response = await sendMessage({
      type: "submit-payload",
      payload,
      settings: {
        workerBaseUrl: normalizeBaseUrl(state.settings.workerBaseUrl),
        authToken: state.settings.authToken.trim()
      }
    });
    if (!response) {
      state.submission.status = "error";
      state.submission.error = { message: "No response from background worker." };
    } else if (response.error) {
      state.submission.status = "error";
      state.submission.error = response.error;
      state.submission.requestId = response.requestId || null;
      if (response.error.fields) {
        state.backendErrors = { ...response.error.fields };
        state.showErrors = true;
      }
    } else if (response.result) {
      state.submission.requestId = response.requestId || response.result.requestId || null;
      state.backendErrors = {};
    }
    render();
  }
  function buildPayload({ candidateId } = {}) {
    const includeMessages = state.form.extracted.includeMessages && Array.isArray(state.form.extracted.messages) && state.form.extracted.messages.length > 0;
    const payload = {
      onBehalfOf: parseInt(state.settings.onBehalfOf, 10),
      outreach: {
        sourceSurface: state.form.outreach.sourceSurface || "",
        linkedinUrl: state.form.outreach.linkedinUrl || "",
        participants: state.form.outreach.participants ? state.form.outreach.participants.split(",").map((value) => value.trim()).filter(Boolean) : [],
        messageSnippet: state.form.outreach.messageSnippet || "",
        messages: includeMessages ? state.form.extracted.messages : []
      },
      application: {
        jobId: parseInt(state.form.application.jobId, 10),
        sourceId: parseInt(state.form.application.sourceId, 10),
        recruiterId: parseInt(state.form.application.recruiterId, 10)
      }
    };
    if (candidateId) {
      payload.candidateId = candidateId;
      return payload;
    }
    const candidate = {
      firstName: state.form.candidate.firstName.trim(),
      lastName: state.form.candidate.lastName.trim(),
      headline: state.form.candidate.headline.trim(),
      company: state.form.candidate.company.trim(),
      location: state.form.candidate.location.trim(),
      emails: toAddressList(state.form.candidate.email),
      phones: toPhoneList(state.form.candidate.phone)
    };
    payload.candidate = candidate;
    return payload;
  }
  function toAddressList(value) {
    if (!value) {
      return [];
    }
    return [{ value: value.trim(), type: "personal" }];
  }
  function toPhoneList(value) {
    if (!value) {
      return [];
    }
    return [{ value: value.trim(), type: "mobile" }];
  }
  function validateForm() {
    const errors = {};
    const isRetryingApplication = Boolean(state.submission.result?.candidateId) && (state.submission.status === "partial" || state.submission.status === "in_progress");
    if (!isRetryingApplication) {
      if (!state.form.candidate.firstName.trim()) {
        errors["candidate.firstName"] = "First name is required.";
      }
      if (!state.form.candidate.lastName.trim()) {
        errors["candidate.lastName"] = "Last name is required.";
      }
    }
    if (!state.form.application.jobId) {
      errors["application.jobId"] = "Select a job.";
    }
    if (!state.form.application.sourceId) {
      errors["application.sourceId"] = "Select a source.";
    }
    if (!state.form.application.recruiterId) {
      errors["application.recruiterId"] = "Select a recruiter.";
    }
    if (!state.form.outreach.linkedinUrl) {
      errors["outreach.linkedinUrl"] = "LinkedIn URL is required.";
    }
    if (!state.form.outreach.sourceSurface.trim()) {
      errors["outreach.sourceSurface"] = "Source surface is required.";
    }
    if (!state.settings.workerBaseUrl.trim()) {
      errors["settings.workerBaseUrl"] = "Add the worker base URL.";
    }
    if (!state.settings.authToken.trim()) {
      errors["settings.authToken"] = "Add the worker auth token.";
    }
    if (!state.settings.onBehalfOf) {
      errors["settings.onBehalfOf"] = "Select an on-behalf-of user.";
    }
    if (!navigator.onLine) {
      errors["offline"] = "You are offline. Reconnect to submit.";
    }
    if (state.reference.status === "error") {
      errors["reference"] = "Reference data failed to load.";
    }
    if (state.reference.status === "ready") {
      if (!state.reference.jobs.length) {
        errors["reference.jobs"] = "No jobs available.";
      }
      if (!state.reference.sources.length) {
        errors["reference.sources"] = "No sources available.";
      }
    }
    if (!state.extraction.supported || !state.extraction.loggedIn) {
      errors["extraction"] = state.extraction.loggedIn ? "Active tab is unsupported." : "LinkedIn login required.";
    }
    const mergedErrors = { ...errors, ...state.backendErrors };
    state.validationErrors = mergedErrors;
    return Object.keys(mergedErrors).length === 0;
  }
  function render() {
    validateForm();
    renderStatus();
    renderErrors();
    renderReferenceData();
    renderSubmission();
    renderInputs();
    renderDetectedLists();
    renderMessages();
    renderButtons();
    renderSettingsSummary();
  }
  function renderSettingsSummary() {
    const summary = getSettingsSummary(state.settings, state.reference);
    dom.settingsSummary.textContent = summary.text;
    dom.settingsSummary.classList.toggle("hidden", !summary.visible);
  }
  function renderStatus() {
    const status = state.extraction.status;
    const warning = status === "warning";
    const ready = status === "ready";
    const idle = status === "idle";
    dom.statusBadge.className = "badge";
    if (idle) {
      dom.statusBadge.classList.add("badge--idle");
      dom.statusBadge.textContent = "Idle";
      dom.statusText.textContent = "Open LinkedIn to get started";
    } else if (warning && !state.extraction.loggedIn) {
      dom.statusBadge.classList.add("badge--waiting");
      dom.statusBadge.textContent = "Action needed";
      dom.statusText.textContent = "Please sign in to LinkedIn";
    } else if (warning && !state.extraction.supported) {
      dom.statusBadge.classList.add("badge--waiting");
      dom.statusBadge.textContent = "Waiting";
      dom.statusText.textContent = "Navigate to a profile or thread";
    } else if (warning) {
      dom.statusBadge.textContent = "Partial";
      dom.statusText.textContent = "Extraction incomplete";
    } else if (ready) {
      dom.statusBadge.classList.add("badge--ready");
      dom.statusBadge.textContent = "Ready";
      dom.statusText.textContent = "Extraction ready";
    } else if (status === "loading") {
      dom.statusBadge.textContent = "Loading";
      dom.statusText.textContent = "Scanning LinkedIn tab";
    } else {
      dom.statusBadge.textContent = "Idle";
      dom.statusText.textContent = "Waiting for extraction";
    }
    dom.statusDetail.textContent = state.extraction.reason || "";
    dom.statusAlerts.innerHTML = "";
    const alerts = [];
    if (!navigator.onLine) {
      alerts.push("Offline: reconnect to submit.");
    }
    if (state.extraction.missing.length) {
      alerts.push(`Missing: ${state.extraction.missing.join(", ")}.`);
    }
    if (state.reference.status === "error") {
      alerts.push("Reference data failed. Retry when ready.");
    }
    if (state.reference.status === "ready" && (!state.reference.jobs.length || !state.reference.sources.length)) {
      alerts.push("Job or source list is empty. Check settings.");
    }
    alerts.forEach((message) => {
      const item = document.createElement("div");
      item.className = "muted";
      item.textContent = message;
      dom.statusAlerts.appendChild(item);
    });
  }
  function renderErrors() {
    const errors = state.validationErrors || {};
    dom.errorList.innerHTML = "";
    const entries = Object.entries(errors);
    const showErrors = state.showErrors || state.submission.status === "error";
    dom.errorSummary.classList.toggle("hidden", !showErrors || entries.length === 0);
    if (showErrors) {
      entries.forEach(([, message]) => {
        const item = document.createElement("li");
        item.textContent = message;
        dom.errorList.appendChild(item);
      });
    }
    dom.errorFields.forEach((element) => {
      const key = element.dataset.errorFor;
      element.textContent = showErrors && key ? errors[key] || "" : "";
    });
  }
  function renderReferenceData() {
    if (state.reference.status === "loading") {
      setSelectLoading(dom.jobSelect, "Loading...");
      setSelectLoading(dom.sourceSelect, "Loading...");
      setSelectLoading(dom.recruiterSelect, "Loading...");
      setSelectLoading(dom.onBehalfOfSelect, "Loading...");
      dom.referenceStatus.textContent = "Connecting...";
      return;
    }
    if (state.reference.status === "error") {
      setSelectLoading(dom.jobSelect, "\u2014");
      setSelectLoading(dom.sourceSelect, "\u2014");
      setSelectLoading(dom.recruiterSelect, "\u2014");
      setSelectLoading(dom.onBehalfOfSelect, "\u2014");
      dom.referenceStatus.textContent = state.reference.error || "Connection failed";
      return;
    }
    if (state.reference.status === "ready") {
      renderSelectOptions(dom.jobSelect, state.reference.jobs, "Select job", "name", "id", false, state.form.application.jobId);
      renderSourceOptions(dom.sourceSelect, state.reference.sources, "Select source", state.form.application.sourceId);
      renderSelectOptions(dom.recruiterSelect, state.reference.users, "Select recruiter", "name", "id", false, state.form.application.recruiterId);
      renderSelectOptions(dom.onBehalfOfSelect, state.reference.users, "Select on-behalf-of user", "name", "id", false, state.settings.onBehalfOf);
      dom.referenceStatus.textContent = "Connected";
    } else {
      setSelectLoading(dom.jobSelect, "\u2014");
      setSelectLoading(dom.sourceSelect, "\u2014");
      setSelectLoading(dom.recruiterSelect, "\u2014");
      setSelectLoading(dom.onBehalfOfSelect, "\u2014");
      dom.referenceStatus.textContent = "Not connected";
    }
  }
  function renderSelectOptions(select, items, placeholder, labelKey, valueKey, includeBlank = false, selectedValue = "") {
    select.innerHTML = "";
    const placeholderOption = document.createElement("option");
    placeholderOption.value = "";
    placeholderOption.textContent = placeholder;
    placeholderOption.disabled = true;
    select.appendChild(placeholderOption);
    if (includeBlank) {
      const emptyOption = document.createElement("option");
      emptyOption.value = "";
      emptyOption.textContent = "(Unassigned)";
      select.appendChild(emptyOption);
    }
    const sortedItems = [...items].sort((a, b) => {
      const aLabel = String(a[labelKey] || "").toLowerCase();
      const bLabel = String(b[labelKey] || "").toLowerCase();
      return aLabel.localeCompare(bLabel);
    });
    sortedItems.forEach((item) => {
      const option = document.createElement("option");
      option.value = String(item[valueKey]);
      option.textContent = String(item[labelKey]);
      select.appendChild(option);
    });
    select.disabled = items.length === 0;
    select.value = selectedValue ? String(selectedValue) : "";
  }
  function renderSourceOptions(select, items, placeholder, selectedValue = "") {
    select.innerHTML = "";
    const placeholderOption = document.createElement("option");
    placeholderOption.value = "";
    placeholderOption.textContent = placeholder;
    placeholderOption.disabled = true;
    select.appendChild(placeholderOption);
    const linkedInSources = items.filter((s) => s.name.toLowerCase().includes("linkedin"));
    const otherSources = items.filter((s) => !s.name.toLowerCase().includes("linkedin"));
    const sortByName = (a, b) => String(a.name || "").toLowerCase().localeCompare(String(b.name || "").toLowerCase());
    linkedInSources.sort(sortByName);
    otherSources.sort(sortByName);
    linkedInSources.forEach((item) => {
      const option = document.createElement("option");
      option.value = String(item.id);
      option.textContent = item.name;
      select.appendChild(option);
    });
    if (linkedInSources.length && otherSources.length) {
      const divider = document.createElement("option");
      divider.disabled = true;
      divider.textContent = "\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500";
      select.appendChild(divider);
    }
    otherSources.forEach((item) => {
      const option = document.createElement("option");
      option.value = String(item.id);
      option.textContent = item.name;
      select.appendChild(option);
    });
    select.disabled = items.length === 0;
    select.value = selectedValue ? String(selectedValue) : "";
  }
  function setSelectLoading(select, label) {
    select.innerHTML = "";
    const option = document.createElement("option");
    option.value = "";
    option.textContent = label;
    select.appendChild(option);
    select.disabled = true;
  }
  function renderInputs() {
    dom.fieldInputs.forEach((input) => {
      const key = input.dataset.field;
      if (key) {
        const value = getFieldValue(key);
        if (input.value !== value) {
          input.value = value;
        }
      }
    });
  }
  function renderDetectedLists() {
    dom.emailOptions.innerHTML = "";
    dom.phoneOptions.innerHTML = "";
    dom.emailDetected.textContent = formatDetected(state.form.extracted.emails.map((item) => item.value));
    dom.phoneDetected.textContent = formatDetected(state.form.extracted.phones.map((item) => item.value));
    state.form.extracted.emails.forEach((email) => {
      const option = document.createElement("option");
      option.value = email.value;
      dom.emailOptions.appendChild(option);
    });
    state.form.extracted.phones.forEach((phone) => {
      const option = document.createElement("option");
      option.value = phone.value;
      dom.phoneOptions.appendChild(option);
    });
  }
  function renderMessages() {
    const messages = state.form.extracted.messages || [];
    dom.messageList.innerHTML = "";
    if (!messages.length) {
      dom.messageEmpty.textContent = "No recent messages detected yet.";
    } else {
      const countLabel = messages.length === 1 ? "1 message captured." : `${messages.length} messages captured.`;
      dom.messageEmpty.textContent = countLabel;
    }
    messages.forEach((message) => {
      const item = document.createElement("div");
      item.className = "message-item";
      const meta = document.createElement("div");
      meta.className = "message-meta";
      meta.textContent = formatMessageMeta(message);
      const body = document.createElement("div");
      body.className = "message-body";
      body.textContent = message.text || "";
      item.appendChild(meta);
      item.appendChild(body);
      dom.messageList.appendChild(item);
    });
    dom.includeMessages.checked = Boolean(state.form.extracted.includeMessages);
    dom.includeMessages.disabled = messages.length === 0;
    dom.messageList.classList.toggle("message-list--disabled", !state.form.extracted.includeMessages);
  }
  function formatMessageMeta(message) {
    const directionLabel = message.direction === "outbound" ? "You" : message.sender ? message.sender : "Them";
    const timestamp = message.timestamp ? ` - ${message.timestamp}` : "";
    return `${directionLabel}${timestamp}`;
  }
  function formatDetected(values) {
    if (!values.length) {
      return "No contact details detected yet.";
    }
    return `Detected: ${values.join(", ")}`;
  }
  function renderSubmission() {
    if (state.submission.status === "in_progress") {
      dom.submissionStatus.textContent = "Submitting to Greenhouse... You can close the panel and check back later.";
    } else if (state.submission.status === "success") {
      dom.submissionStatus.textContent = "Submission complete.";
    } else if (state.submission.status === "partial") {
      const requestId = state.submission.requestId ? ` (Request ID: ${state.submission.requestId})` : "";
      const message = state.submission.error?.message || "Candidate created, application pending.";
      dom.submissionStatus.textContent = `${message}${requestId}`;
    } else if (state.submission.status === "error") {
      const requestId = state.submission.requestId ? ` (Request ID: ${state.submission.requestId})` : "";
      let message = state.submission.error?.message || "Submission failed.";
      if (state.submission.error?.code === "unauthorized") {
        message = "Unauthorized. Check the worker auth token or contact support.";
      }
      dom.submissionStatus.textContent = `${message}${requestId}`;
    } else {
      dom.submissionStatus.textContent = "";
    }
    renderResultLinks();
  }
  function renderResultLinks() {
    const result = state.submission.result;
    const canShow = state.submission.status === "success" || state.submission.status === "partial";
    if (!canShow || !result) {
      dom.resultCard.classList.add("hidden");
      return;
    }
    dom.resultLinks.innerHTML = "";
    const links = [];
    if (result.candidateUrl) {
      links.push({ label: "Open candidate", url: result.candidateUrl });
    }
    if (result.applicationUrl) {
      links.push({ label: "Open application", url: result.applicationUrl });
    }
    if (!links.length) {
      dom.resultCard.classList.add("hidden");
      return;
    }
    links.forEach((link) => {
      const anchor = document.createElement("a");
      anchor.href = link.url;
      anchor.target = "_blank";
      anchor.rel = "noreferrer";
      anchor.textContent = link.label;
      dom.resultLinks.appendChild(anchor);
    });
    dom.resultCard.classList.remove("hidden");
  }
  function renderButtons() {
    const extractionIncomplete = !state.extraction.supported || !state.extraction.loggedIn;
    const referenceReady = state.reference.status === "ready" && state.reference.jobs.length > 0 && state.reference.sources.length > 0;
    const settingsReady = state.settings.workerBaseUrl.trim() && state.settings.authToken.trim() && state.settings.onBehalfOf;
    const hasErrors = Object.keys(state.validationErrors || {}).length > 0;
    const isPartial = state.submission.status === "partial";
    dom.submitButton.disabled = extractionIncomplete || !referenceReady || !settingsReady || !navigator.onLine || state.submission.status === "in_progress" || hasErrors || isPartial;
    dom.retryReference.disabled = state.reference.status === "loading";
    dom.retryApplication.classList.toggle("hidden", !isPartial);
    dom.retryApplication.disabled = !isPartial || state.submission.status === "in_progress" || hasErrors;
  }
  function getFieldValue(key) {
    const [section, field] = key.split(".");
    if (section === "settings") {
      return state.settings[field] || "";
    }
    if (section === "candidate") {
      return state.form.candidate[field] || "";
    }
    if (section === "outreach") {
      return state.form.outreach[field] || "";
    }
    if (section === "application") {
      return state.form.application[field] || "";
    }
    return "";
  }
  function sendMessage(message) {
    return new Promise((resolve) => {
      chrome.runtime.sendMessage(message, (response) => {
        if (chrome.runtime.lastError) {
          resolve({ error: chrome.runtime.lastError.message });
          return;
        }
        resolve(response);
      });
    });
  }
})();
//# sourceMappingURL=panel.js.map
