"use strict";
(() => {
  // src/helpers/linkedin-extraction.ts
  function splitName(fullName) {
    const normalized = (fullName || "").trim();
    if (!normalized) {
      return { fullName: "", firstName: "", lastName: "" };
    }
    const tokens = normalized.split(/\s+/);
    if (tokens.length === 1) {
      return { fullName: normalized, firstName: tokens[0], lastName: "" };
    }
    return {
      fullName: normalized,
      firstName: tokens[0],
      lastName: tokens.slice(1).join(" ")
    };
  }
  function isMessagingThread(url) {
    return url.pathname.startsWith("/messaging/thread");
  }
  function getLimitedProfileReason({ nameText, headline, company, location, missingFields }) {
    if (!missingFields || missingFields.length === 0) {
      return "";
    }
    const normalizedName = (nameText || "").trim();
    if (!normalizedName || /linkedin member/i.test(normalizedName)) {
      return "Profile visibility is limited. Some details are hidden.";
    }
    const missingCoreCount = [headline, company, location].filter((value) => !value || !value.trim()).length;
    if (missingCoreCount >= 2) {
      return "Some profile details are unavailable. The profile may be private or still loading.";
    }
    return "";
  }
  function getText(selectors) {
    for (const selector of selectors) {
      const element = document.querySelector(selector);
      if (element && element.textContent) {
        const text = element.textContent.trim();
        if (text) {
          return text;
        }
      }
    }
    return "";
  }
  function getTextFrom(root, selectors) {
    for (const selector of selectors) {
      const element = root.querySelector(selector);
      if (element && element.textContent) {
        const text = element.textContent.trim();
        if (text) {
          return text;
        }
      }
    }
    return "";
  }
  function extractEmails() {
    const links = Array.from(document.querySelectorAll("a[href^='mailto:']"));
    const emails = links.map((link) => link.getAttribute("href")?.replace("mailto:", "").split("?")[0]).filter((email) => Boolean(email));
    return uniqueValues(emails).map((value) => ({ value, type: "personal" }));
  }
  function extractPhones() {
    const links = Array.from(document.querySelectorAll("a[href^='tel:']"));
    const phones = links.map((link) => link.getAttribute("href")?.replace("tel:", "").trim()).filter((phone) => Boolean(phone));
    return uniqueValues(phones).map((value) => ({ value, type: "mobile" }));
  }
  function extractParticipants() {
    const rawText = getText([
      "h2.msg-thread__title",
      "h2.msg-thread__title span",
      "div.msg-thread__participants",
      "header h2"
    ]);
    if (!rawText) {
      return [];
    }
    const cleaned = rawText.replace(/\n/g, " ").trim();
    return cleaned.split(/,|&| and /).map((value) => value.trim()).filter(Boolean);
  }
  function extractMessages() {
    const groups = Array.from(document.querySelectorAll(".msg-s-message-group, .msg-s-message-list__event"));
    const messages = [];
    groups.forEach((group) => {
      const isOutbound = isMessageOutbound(group);
      const sender = getTextFrom(group, [
        ".msg-s-message-group__profile-link",
        ".msg-s-message-group__name",
        "span.msg-s-message-group__profile-link"
      ]);
      const timestamp = getTextFrom(group, [
        "time",
        ".msg-s-message-group__timestamp",
        ".msg-s-message-group__timestamp span"
      ]);
      const messageNodes = group.querySelectorAll(".msg-s-message-group__message, .msg-s-message-group__message span, .msg-s-message-group__message p");
      messageNodes.forEach((node) => {
        const text = node.textContent?.trim();
        if (!text) {
          return;
        }
        messages.push({
          direction: isOutbound ? "outbound" : "inbound",
          text,
          timestamp: timestamp || "",
          sender: sender || ""
        });
      });
    });
    return messages.slice(-8);
  }
  function isMessageOutbound(group) {
    const className = group.className || "";
    if (className.includes("msg-s-message-group--self") || className.includes("msg-s-message-group--sent")) {
      return true;
    }
    return false;
  }
  function uniqueValues(values) {
    return Array.from(new Set(values));
  }
  function buildMissingFields(candidate, outreach) {
    const missing = [];
    if (!candidate.firstName) {
      missing.push("first name");
    }
    if (!candidate.lastName) {
      missing.push("last name");
    }
    if (!candidate.headline) {
      missing.push("headline");
    }
    if (!candidate.company) {
      missing.push("company");
    }
    if (!candidate.location) {
      missing.push("location");
    }
    if (!outreach.linkedinUrl) {
      missing.push("linkedin url");
    }
    return missing;
  }
  function getCompanyFromExperience() {
    const experienceSection = document.querySelector("section#experience, section[id*='experience']");
    if (!experienceSection) {
      return "";
    }
    const role = experienceSection.querySelector("li.pvs-list__paged-list-item, li");
    if (!role) {
      return "";
    }
    const company = getTextFrom(role, [
      "span.t-14.t-normal",
      "span.t-14",
      "span[aria-hidden='true']"
    ]);
    return company;
  }
  function isLoggedIn() {
    return !document.querySelector("input[name='session_key'], form[action*='login'], .sign-in-form");
  }

  // src/content-script.ts
  chrome.runtime.onMessage.addListener(
    (message, _sender, sendResponse) => {
      if (!message || message.type !== "extract") {
        return;
      }
      sendResponse(extractLinkedInData());
    }
  );
  function extractLinkedInData() {
    const url = new URL(window.location.href);
    const loggedIn = isLoggedIn();
    if (!loggedIn) {
      return {
        supported: false,
        loggedIn: false,
        reason: "LinkedIn login required"
      };
    }
    if (url.pathname.startsWith("/in/")) {
      return extractProfile(url);
    }
    if (isMessagingThread(url)) {
      return extractMessaging(url);
    }
    if (url.pathname.startsWith("/messaging")) {
      return {
        supported: false,
        loggedIn: true,
        reason: "Open a LinkedIn messaging thread to extract data."
      };
    }
    return {
      supported: false,
      loggedIn: true,
      reason: "Unsupported LinkedIn page"
    };
  }
  function extractProfile(url) {
    const nameText = getText([
      "h1.text-heading-xlarge",
      "h1"
    ]);
    const headline = getText([
      "div.text-body-medium.break-words",
      "div.text-body-medium"
    ]);
    const location = getText([
      "span.text-body-small.inline",
      "span.text-body-small"
    ]);
    const company = getCompanyFromExperience();
    const emails = extractEmails();
    const phones = extractPhones();
    const nameParts = splitName(nameText);
    const candidate = {
      fullName: nameParts.fullName,
      firstName: nameParts.firstName,
      lastName: nameParts.lastName,
      headline,
      company,
      location,
      emails,
      phones
    };
    const outreach = {
      sourceSurface: "profile",
      linkedinUrl: url.toString(),
      participants: nameParts.fullName ? [nameParts.fullName] : [],
      messages: [],
      messageSnippet: ""
    };
    const missingFields = buildMissingFields(candidate, outreach);
    const visibilityNotice = getLimitedProfileReason({
      nameText,
      headline,
      company,
      location,
      missingFields
    });
    return {
      supported: true,
      loggedIn: true,
      candidate,
      outreach,
      missingFields,
      partial: missingFields.length > 0,
      reason: visibilityNotice
    };
  }
  function extractMessaging(url) {
    const participants = extractParticipants();
    const messages = extractMessages();
    const candidateName = participants.length === 1 ? participants[0] : "";
    const nameParts = splitName(candidateName);
    const candidate = {
      fullName: nameParts.fullName,
      firstName: nameParts.firstName,
      lastName: nameParts.lastName,
      headline: "",
      company: "",
      location: "",
      emails: extractEmails(),
      phones: extractPhones()
    };
    const outreach = {
      sourceSurface: "messaging",
      linkedinUrl: url.toString(),
      participants,
      messages,
      messageSnippet: ""
    };
    const missingFields = buildMissingFields(candidate, outreach);
    return {
      supported: true,
      loggedIn: true,
      candidate,
      outreach,
      missingFields,
      partial: missingFields.length > 0
    };
  }
})();
//# sourceMappingURL=content-script.js.map
