"use strict";
(() => {
  // src/helpers/url-validation.ts
  function isLinkedInUrl(tabUrl) {
    if (!tabUrl) {
      return false;
    }
    try {
      const parsed = new URL(tabUrl);
      return parsed.hostname === "www.linkedin.com" || parsed.hostname === "linkedin.com";
    } catch {
      return false;
    }
  }
  function isSupportedLinkedInUrl(tabUrl) {
    if (!tabUrl) {
      return false;
    }
    let parsed;
    try {
      parsed = new URL(tabUrl);
    } catch {
      return false;
    }
    if (parsed.hostname !== "www.linkedin.com" && parsed.hostname !== "linkedin.com") {
      return false;
    }
    return parsed.pathname.startsWith("/in/") || parsed.pathname.startsWith("/messaging/thread");
  }

  // src/background.ts
  var SUBMISSION_KEY = "submissionState";
  chrome.sidePanel.setPanelBehavior({ openPanelOnActionClick: true });
  chrome.runtime.onMessage.addListener(
    (message, _sender, sendResponse) => {
      if (!message || typeof message !== "object") {
        sendResponse({ error: "Invalid message." });
        return;
      }
      if (message.type === "request-extraction") {
        requestExtraction().then(sendResponse);
        return true;
      }
      if (message.type === "submit-payload") {
        submitPayload(message.payload, message.settings).then(sendResponse);
        return true;
      }
      if (message.type === "get-submission-state") {
        getSubmissionState().then((state) => sendResponse({ state }));
        return true;
      }
      sendResponse({ error: "Unknown message type." });
    }
  );
  chrome.tabs.onUpdated.addListener((tabId, changeInfo) => {
    if (changeInfo.status === "complete" || changeInfo.url) {
      chrome.runtime.sendMessage({ type: "tab-changed", tabId, url: changeInfo.url || "" });
    }
  });
  chrome.tabs.onActivated.addListener(() => {
    chrome.runtime.sendMessage({ type: "tab-changed" });
  });
  async function requestExtraction() {
    const tab = await getActiveTab();
    if (!tab || !tab.id) {
      return { supported: false, loggedIn: true, reason: "No active tab." };
    }
    const isLinkedIn = isLinkedInUrl(tab.url);
    const supportsExtraction = isSupportedLinkedInUrl(tab.url);
    if (!supportsExtraction) {
      return {
        supported: false,
        loggedIn: true,
        onLinkedIn: isLinkedIn,
        reason: isLinkedIn ? "Navigate to a profile or messaging thread." : "Open LinkedIn to get started."
      };
    }
    return new Promise((resolve) => {
      const timeout = setTimeout(() => {
        resolve({ supported: true, loggedIn: true, partial: true, reason: "Extraction timed out." });
      }, 2e3);
      chrome.tabs.sendMessage(tab.id, { type: "extract" }, (response) => {
        clearTimeout(timeout);
        if (chrome.runtime.lastError) {
          resolve({
            error: chrome.runtime.lastError.message,
            supported: true,
            loggedIn: true
          });
          return;
        }
        if (!response) {
          resolve({
            error: "No response from LinkedIn page.",
            supported: true,
            loggedIn: true
          });
          return;
        }
        resolve(response);
      });
    });
  }
  async function getActiveTab() {
    const tabs = await chrome.tabs.query({ active: true, currentWindow: true });
    return tabs[0];
  }
  async function submitPayload(payload, settings) {
    if (!settings?.workerBaseUrl || !settings?.authToken) {
      return { error: { message: "Worker settings are required." } };
    }
    const submissionState = {
      status: "in_progress",
      startedAt: (/* @__PURE__ */ new Date()).toISOString(),
      error: null,
      result: null
    };
    await setSubmissionState(submissionState);
    try {
      const response = await fetch(`${settings.workerBaseUrl}/api/candidates`, {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
          Authorization: `Bearer ${settings.authToken}`
        },
        body: JSON.stringify(payload)
      });
      const data = await response.json().catch(() => ({}));
      if (!response.ok) {
        const error = data?.error || { message: `Request failed (${response.status})` };
        const nextState2 = {
          status: "error",
          error,
          result: data?.result || null,
          requestId: data?.requestId || null
        };
        await setSubmissionState(nextState2);
        return { error, requestId: data?.requestId || null };
      }
      const isPartial = Boolean(data?.partial);
      const nextState = {
        status: isPartial ? "partial" : "success",
        error: isPartial ? data?.applicationError || { message: "Candidate created, application pending." } : null,
        result: data,
        requestId: data?.requestId || null
      };
      await setSubmissionState(nextState);
      return { result: data, requestId: data?.requestId || null };
    } catch (error) {
      const nextState = {
        status: "error",
        error: { message: error.message || "Network error" },
        result: null
      };
      await setSubmissionState(nextState);
      return { error: nextState.error };
    }
  }
  async function setSubmissionState(state) {
    await chrome.storage.local.set({ [SUBMISSION_KEY]: state });
    chrome.runtime.sendMessage({ type: "submission-updated", state });
  }
  async function getSubmissionState() {
    const data = await chrome.storage.local.get([SUBMISSION_KEY]);
    return data[SUBMISSION_KEY] || { status: "idle", error: null, result: null };
  }
})();
//# sourceMappingURL=background.js.map
